<?php
declare(strict_types=1);

require_once __DIR__ . '/../src/response.php';
require_once __DIR__ . '/../src/helpers.php';
require_once __DIR__ . '/../src/db.php';
require_once __DIR__ . '/../src/jwt.php';
require_once __DIR__ . '/../src/auth.php';

$config = require __DIR__ . '/../src/config.php';

$method = strtoupper($_SERVER['REQUEST_METHOD'] ?? 'GET');
$origin = $_SERVER['HTTP_ORIGIN'] ?? '';
applyCors($origin, $config['app']['allowed_origins']);

try {
    $pdo = db($config['db']);
} catch (Throwable $exception) {
    fail('Database connection failed', 500, ['details' => $exception->getMessage()]);
    return;
}

$path = resolveApiPath();

function makeApiRequest(string $url, array $headers = []): array
{
    $ch = curl_init($url);
    if ($ch === false) {
        throw new RuntimeException('Unable to initialize external request');
    }

    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 10,
        CURLOPT_HTTPHEADER => $headers,
    ]);

    $response = curl_exec($ch);
    if ($response === false) {
        $error = curl_error($ch);
        curl_close($ch);
        throw new RuntimeException($error !== '' ? $error : 'External request failed');
    }

    $statusCode = (int) curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($statusCode >= 400) {
        throw new RuntimeException('External request failed with status ' . $statusCode);
    }

    $decoded = json_decode($response, true);
    if (!is_array($decoded)) {
        throw new RuntimeException('Invalid external API response');
    }

    return $decoded;
}

function getApiKeys(PDO $pdo, array $envApiKeys, bool $refresh = false): array
{
    static $cache = null;

    if ($refresh || $cache === null) {
        $cache = $envApiKeys;
        try {
            $stmt = $pdo->query('SELECT name, value FROM api_keys');
            if ($stmt !== false) {
                foreach ($stmt as $row) {
                    $cache[$row['name']] = $row['value'];
                }
            }
        } catch (Throwable $exception) {
            // Fall back to environment values if DB read fails.
        }
    }

    return $cache;
}

function getApiKey(PDO $pdo, array $envApiKeys, string $name): ?string
{
    $keys = getApiKeys($pdo, $envApiKeys);
    $value = $keys[$name] ?? null;
    if ($value === null || $value === '') {
        return null;
    }
    return (string) $value;
}

function isoNow(int $offsetSeconds = 0): string
{
    return gmdate('Y-m-d\TH:i:s\Z', time() + $offsetSeconds);
}

function parseFloatQuery($value): ?float
{
    if ($value === null || $value === '') {
        return null;
    }

    if (!is_numeric($value)) {
        return null;
    }

    return (float) $value;
}

function parseIntQuery($value): ?int
{
    if ($value === null || $value === '') {
        return null;
    }

    if (!is_numeric($value)) {
        return null;
    }

    return (int) $value;
}

function clampFloat(float $value, float $min, float $max): float
{
    return min($max, max($min, $value));
}

function clampInt(int $value, int $min, int $max): int
{
    return min($max, max($min, $value));
}

function toRadians(float $degrees): float
{
    return ($degrees * M_PI) / 180.0;
}

function haversineDistanceNm(float $lat1, float $lon1, float $lat2, float $lon2): float
{
    $earthRadiusKm = 6371.0;
    $dLat = toRadians($lat2 - $lat1);
    $dLon = toRadians($lon2 - $lon1);

    $a = sin($dLat / 2) ** 2
        + cos(toRadians($lat1)) * cos(toRadians($lat2)) * sin($dLon / 2) ** 2;

    $c = 2 * atan2(sqrt($a), sqrt(1 - $a));
    $distanceKm = $earthRadiusKm * $c;
    return $distanceKm * 0.539956803;
}

function bearingDegrees(float $lat1, float $lon1, float $lat2, float $lon2): float
{
    $dLon = toRadians($lon2 - $lon1);
    $y = sin($dLon) * cos(toRadians($lat2));
    $x = cos(toRadians($lat1)) * sin(toRadians($lat2))
        - sin(toRadians($lat1)) * cos(toRadians($lat2)) * cos($dLon);

    $bearing = rad2deg(atan2($y, $x));
    return fmod(($bearing + 360.0), 360.0);
}

function geometryCenter(?array $geometry): ?array
{
    if (!$geometry || !isset($geometry['type']) || !isset($geometry['coordinates']) || !is_array($geometry['coordinates'])) {
        return null;
    }

    $type = $geometry['type'];
    $coordinates = $geometry['coordinates'];

    if ($type === 'Point' && isset($coordinates[0], $coordinates[1])) {
        $lon = (float) $coordinates[0];
        $lat = (float) $coordinates[1];
        return ['lat' => $lat, 'lon' => $lon];
    }

    if ($type === 'Polygon' && isset($coordinates[0]) && is_array($coordinates[0])) {
        $ring = $coordinates[0];
        $sumLat = 0.0;
        $sumLon = 0.0;
        $count = 0;
        foreach ($ring as $point) {
            if (is_array($point) && isset($point[0], $point[1])) {
                $sumLon += (float) $point[0];
                $sumLat += (float) $point[1];
                $count++;
            }
        }
        if ($count === 0) {
            return null;
        }
        return ['lat' => $sumLat / $count, 'lon' => $sumLon / $count];
    }

    if (
        $type === 'MultiPolygon'
        && isset($coordinates[0], $coordinates[0][0])
        && is_array($coordinates[0][0])
    ) {
        $ring = $coordinates[0][0];
        $sumLat = 0.0;
        $sumLon = 0.0;
        $count = 0;
        foreach ($ring as $point) {
            if (is_array($point) && isset($point[0], $point[1])) {
                $sumLon += (float) $point[0];
                $sumLat += (float) $point[1];
                $count++;
            }
        }
        if ($count === 0) {
            return null;
        }
        return ['lat' => $sumLat / $count, 'lon' => $sumLon / $count];
    }

    return null;
}

function fetchOpenAipCore(PDO $pdo, array $envApiKeys, string $endpoint, array $queryParams = []): array
{
    $openAipKey = getApiKey($pdo, $envApiKeys, 'openaip');
    if ($openAipKey === null) {
        throw new RuntimeException('OpenAIP API key missing');
    }

    $normalized = [];
    foreach ($queryParams as $key => $value) {
        if ($value === null || $value === '') {
            continue;
        }
        $normalized[$key] = (string) $value;
    }

    $queryString = http_build_query($normalized);
    $cacheTtlSeconds = 60;
    $cacheDir = rtrim(sys_get_temp_dir(), DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . 'weatherly-openaip-cache';
    $cacheKey = sha1($endpoint . '?' . $queryString);
    $cacheFile = $cacheDir . DIRECTORY_SEPARATOR . $cacheKey . '.json';

    if (!is_dir($cacheDir)) {
        @mkdir($cacheDir, 0775, true);
    }

    if (is_file($cacheFile) && (time() - (int) filemtime($cacheFile)) <= $cacheTtlSeconds) {
        $cachedRaw = file_get_contents($cacheFile);
        if ($cachedRaw !== false) {
            $cachedData = json_decode($cachedRaw, true);
            if (is_array($cachedData)) {
                return $cachedData;
            }
        }
    }

    $url = 'https://api.core.openaip.net/api' . $endpoint;
    if ($queryString !== '') {
        $url .= '?' . $queryString;
    }

    $data = makeApiRequest($url, ['x-openaip-api-key: ' . $openAipKey]);
    @file_put_contents($cacheFile, (string) json_encode($data, JSON_UNESCAPED_SLASHES));
    return $data;
}

function defaultPlatformSettings(): array
{
    return [
        'maintenanceMode' => false,
        'allowRegistrations' => true,
        'defaultAirport' => 'KJFK',
        'supportEmail' => 'support@weatherly.co.ke',
        'pirepAutoApprove' => true,
    ];
}

function normalizeBoolSetting($value, bool $default): bool
{
    if (is_bool($value)) {
        return $value;
    }

    if (is_int($value) || is_float($value)) {
        return ((int) $value) !== 0;
    }

    if (is_string($value)) {
        $normalized = strtolower(trim($value));
        if (in_array($normalized, ['1', 'true', 'yes', 'on'], true)) {
            return true;
        }
        if (in_array($normalized, ['0', 'false', 'no', 'off', ''], true)) {
            return false;
        }
    }

    return $default;
}

function ensurePlatformSettingsTable(PDO $pdo): void
{
    static $ready = false;
    if ($ready) {
        return;
    }

    $pdo->exec(
        'CREATE TABLE IF NOT EXISTS platform_settings (
            name VARCHAR(64) NOT NULL PRIMARY KEY,
            value LONGTEXT NULL,
            updated_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci'
    );

    $ready = true;
}

function readPlatformSettings(PDO $pdo): array
{
    $defaults = defaultPlatformSettings();

    try {
        ensurePlatformSettingsTable($pdo);
    } catch (Throwable $exception) {
        return $defaults;
    }

    $settings = $defaults;
    try {
        $stmt = $pdo->query('SELECT name, value FROM platform_settings');
        $rows = $stmt === false ? [] : $stmt->fetchAll();

        foreach ($rows as $row) {
            $name = (string) ($row['name'] ?? '');
            if ($name === '') {
                continue;
            }

            $raw = $row['value'] ?? null;
            if ($raw === null || $raw === '') {
                $settings[$name] = null;
                continue;
            }

            $decoded = json_decode((string) $raw, true);
            if (json_last_error() === JSON_ERROR_NONE) {
                $settings[$name] = $decoded;
                continue;
            }

            $settings[$name] = $raw;
        }
    } catch (Throwable $exception) {
        return $defaults;
    }

    $settings['maintenanceMode'] = normalizeBoolSetting($settings['maintenanceMode'] ?? null, false);
    $settings['allowRegistrations'] = normalizeBoolSetting($settings['allowRegistrations'] ?? null, true);
    $settings['pirepAutoApprove'] = normalizeBoolSetting($settings['pirepAutoApprove'] ?? null, true);

    $defaultAirport = strtoupper(trim((string) ($settings['defaultAirport'] ?? '')));
    $settings['defaultAirport'] = $defaultAirport !== '' ? $defaultAirport : 'KJFK';

    $supportEmail = trim((string) ($settings['supportEmail'] ?? ''));
    $settings['supportEmail'] = $supportEmail !== '' ? $supportEmail : 'support@weatherly.co.ke';

    return $settings;
}

function savePlatformSettings(PDO $pdo, array $incoming): array
{
    ensurePlatformSettingsTable($pdo);
    $settings = readPlatformSettings($pdo);

    if (array_key_exists('maintenanceMode', $incoming)) {
        $settings['maintenanceMode'] = normalizeBoolSetting($incoming['maintenanceMode'], (bool) $settings['maintenanceMode']);
    }

    if (array_key_exists('allowRegistrations', $incoming)) {
        $settings['allowRegistrations'] = normalizeBoolSetting($incoming['allowRegistrations'], (bool) $settings['allowRegistrations']);
    }

    if (array_key_exists('pirepAutoApprove', $incoming)) {
        $settings['pirepAutoApprove'] = normalizeBoolSetting($incoming['pirepAutoApprove'], (bool) $settings['pirepAutoApprove']);
    }

    if (array_key_exists('defaultAirport', $incoming)) {
        $icao = strtoupper(trim((string) $incoming['defaultAirport']));
        if ($icao !== '') {
            $settings['defaultAirport'] = $icao;
        }
    }

    if (array_key_exists('supportEmail', $incoming)) {
        $supportEmail = trim((string) $incoming['supportEmail']);
        if ($supportEmail !== '') {
            $settings['supportEmail'] = $supportEmail;
        }
    }

    $stmt = $pdo->prepare(
        'INSERT INTO platform_settings (name, value, updated_at)
         VALUES (:name, :value, NOW())
         ON DUPLICATE KEY UPDATE value = VALUES(value), updated_at = NOW()'
    );

    foreach ($settings as $name => $value) {
        $stmt->execute([
            'name' => (string) $name,
            'value' => json_encode($value, JSON_UNESCAPED_SLASHES),
        ]);
    }

    return $settings;
}

// 1. Auth Routes
if ($method === 'POST' && $path === '/auth/register') {
    $payload = jsonBody();
    $email = strtolower(trim((string) ($payload['email'] ?? '')));
    $password = (string) ($payload['password'] ?? '');
    $username = trim((string) ($payload['username'] ?? ''));
    $licenseNumber = trim((string) ($payload['licenseNumber'] ?? ''));
    $pilotCertification = trim((string) ($payload['pilotCertification'] ?? ''));

    if ($email === '' || $password === '') {
        fail('Email and password required', 400);
        return;
    }

    $platformSettings = readPlatformSettings($pdo);
    if (($platformSettings['allowRegistrations'] ?? true) !== true) {
        fail('User registration is currently disabled by platform settings', 403);
        return;
    }

    try {
        $id = generateUuidV4();
        $hashedPassword = password_hash($password, PASSWORD_BCRYPT);
        $preferences = json_encode(defaultPreferences(), JSON_UNESCAPED_SLASHES);
        $favoriteAirports = json_encode([], JSON_UNESCAPED_SLASHES);

        $insert = $pdo->prepare(
            'INSERT INTO users (id, email, password_hash, username, license_number, pilot_certification, favorite_airports, preferences)
             VALUES (:id, :email, :password_hash, :username, :license_number, :pilot_certification, :favorite_airports, :preferences)'
        );

        $insert->execute([
            'id' => $id,
            'email' => $email,
            'password_hash' => $hashedPassword,
            'username' => $username !== '' ? $username : explode('@', $email)[0],
            'license_number' => $licenseNumber !== '' ? $licenseNumber : null,
            'pilot_certification' => $pilotCertification !== '' ? $pilotCertification : null,
            'favorite_airports' => $favoriteAirports,
            'preferences' => $preferences,
        ]);

        $select = $pdo->prepare('SELECT * FROM users WHERE id = :id LIMIT 1');
        $select->execute(['id' => $id]);
        $user = $select->fetch();

        $token = jwtEncode([
            'id' => $id,
            'email' => $email,
            'role' => 'user',
        ], $config['app']['jwt_secret']);

        sendJson(['token' => $token, 'user' => toPublicUser(is_array($user) ? $user : [])]);
    } catch (PDOException $exception) {
        if ($exception->getCode() === '23000') {
            fail('Email already registered', 400);
            return;
        }

        fail('Registration failed', 500, ['details' => $exception->getMessage()]);
    }
    return;
}

if ($method === 'POST' && $path === '/auth/login') {
    $payload = jsonBody();
    $email = strtolower(trim((string) ($payload['email'] ?? '')));
    $password = (string) ($payload['password'] ?? '');

    try {
        $select = $pdo->prepare('SELECT * FROM users WHERE email = :email LIMIT 1');
        $select->execute(['email' => $email]);
        $user = $select->fetch();

        if (!$user || !password_verify($password, (string) $user['password_hash'])) {
            fail('Invalid credentials', 401);
            return;
        }

        $token = jwtEncode([
            'id' => $user['id'],
            'email' => $user['email'],
            'role' => $user['role'] ?? 'user',
        ], $config['app']['jwt_secret']);

        sendJson(['token' => $token, 'user' => toPublicUser(is_array($user) ? $user : [])]);
    } catch (Throwable $exception) {
        fail('Login failed', 500);
    }
    return;
}

if ($method === 'GET' && $path === '/auth/me') {
    $authUser = requireAuth($pdo, $config['app']['jwt_secret']);
    $select = $pdo->prepare('SELECT * FROM users WHERE id = :id LIMIT 1');
    $select->execute(['id' => $authUser['id']]);
    $user = $select->fetch();
    sendJson(['user' => toPublicUser(is_array($user) ? $user : [])]);
    return;
}

if ($method === 'PUT' && $path === '/user/profile') {
    $authUser = requireAuth($pdo, $config['app']['jwt_secret']);
    $payload = jsonBody();

    $select = $pdo->prepare('SELECT * FROM users WHERE id = :id LIMIT 1');
    $select->execute(['id' => $authUser['id']]);
    $current = $select->fetch();

    if (!$current) {
        fail('User not found', 404);
        return;
    }

    if (array_key_exists('favoriteAirports', $payload) && !is_array($payload['favoriteAirports'])) {
        fail('favoriteAirports must be an array', 400);
        return;
    }

    $updatedUsername = array_key_exists('username', $payload) ? (string) $payload['username'] : (string) $current['username'];
    $updatedLicense = array_key_exists('licenseNumber', $payload) ? (string) $payload['licenseNumber'] : (string) ($current['license_number'] ?? '');
    $updatedCertification = array_key_exists('pilotCertification', $payload) ? (string) $payload['pilotCertification'] : (string) ($current['pilot_certification'] ?? '');
    $updatedFavorites = array_key_exists('favoriteAirports', $payload)
        ? json_encode(array_map(static fn ($icao) => strtoupper((string) $icao), $payload['favoriteAirports']), JSON_UNESCAPED_SLASHES)
        : (string) ($current['favorite_airports'] ?? '[]');

    $update = $pdo->prepare(
        'UPDATE users
         SET username = :username, license_number = :license_number, pilot_certification = :pilot_certification, favorite_airports = :favorite_airports
         WHERE id = :id'
    );
    $update->execute([
        'username' => $updatedUsername,
        'license_number' => $updatedLicense !== '' ? $updatedLicense : null,
        'pilot_certification' => $updatedCertification !== '' ? $updatedCertification : null,
        'favorite_airports' => $updatedFavorites,
        'id' => $authUser['id'],
    ]);

    $select->execute(['id' => $authUser['id']]);
    $user = $select->fetch();
    sendJson(['success' => true, 'user' => toPublicUser(is_array($user) ? $user : [])]);
    return;
}

if ($method === 'PUT' && $path === '/user/preferences') {
    $authUser = requireAuth($pdo, $config['app']['jwt_secret']);
    $payload = jsonBody();

    $select = $pdo->prepare('SELECT preferences FROM users WHERE id = :id LIMIT 1');
    $select->execute(['id' => $authUser['id']]);
    $current = $select->fetch();

    $currentPrefs = safeJsonDecode($current['preferences'] ?? null, defaultPreferences());
    $updatedPrefs = array_merge(is_array($currentPrefs) ? $currentPrefs : defaultPreferences(), $payload);

    $update = $pdo->prepare('UPDATE users SET preferences = :preferences WHERE id = :id');
    $update->execute([
        'preferences' => json_encode($updatedPrefs, JSON_UNESCAPED_SLASHES),
        'id' => $authUser['id'],
    ]);

    $userSelect = $pdo->prepare('SELECT * FROM users WHERE id = :id LIMIT 1');
    $userSelect->execute(['id' => $authUser['id']]);
    $user = $userSelect->fetch();

    sendJson(['success' => true, 'user' => toPublicUser(is_array($user) ? $user : [])]);
    return;
}

// 2. Weather Data
if ($method === 'GET' && $path === '/weather/metar') {
    requireAuth($pdo, $config['app']['jwt_secret']);
    $icao = strtoupper(trim((string) ($_GET['icao'] ?? '')));
    if ($icao === '') {
        fail('ICAO required', 400);
        return;
    }

    $checkwxKey = getApiKey($pdo, $config['api_keys'], 'checkwx');
    if ($checkwxKey === null) {
        sendJson([
            'success' => true,
            'code' => 'MOCK',
            'data' => [
                'station' => ['ident' => $icao],
                'raw_text' => $icao . ' MOCK METAR 22/15 A2992',
                'temperature' => ['celsius' => 22],
                'dewpoint' => ['celsius' => 15],
                'flight_category' => 'VFR',
            ],
        ]);
        return;
    }

    try {
        $data = makeApiRequest(
            'https://api.checkwx.com/v1/metar/' . rawurlencode($icao) . '/decoded',
            ['X-API-Key: ' . $checkwxKey]
        );

        sendJson([
            'success' => true,
            'data' => $data['data'][0] ?? null,
        ]);
    } catch (Throwable $exception) {
        fail('Failed to fetch METAR', 500);
    }
    return;
}

if ($method === 'GET' && $path === '/weather/taf') {
    requireAuth($pdo, $config['app']['jwt_secret']);
    $icao = strtoupper(trim((string) ($_GET['icao'] ?? '')));
    if ($icao === '') {
        fail('ICAO required', 400);
        return;
    }

    $checkwxKey = getApiKey($pdo, $config['api_keys'], 'checkwx');
    if ($checkwxKey === null) {
        sendJson([
            'success' => true,
            'code' => 'MOCK',
            'data' => [
                'station' => ['ident' => $icao],
                'raw_text' => 'TAF ' . $icao . ' MOCK',
            ],
        ]);
        return;
    }

    try {
        $data = makeApiRequest(
            'https://api.checkwx.com/v1/taf/' . rawurlencode($icao) . '/decoded',
            ['X-API-Key: ' . $checkwxKey]
        );

        sendJson([
            'success' => true,
            'data' => $data['data'][0] ?? null,
        ]);
    } catch (Throwable $exception) {
        fail('Failed to fetch TAF', 500);
    }
    return;
}

if ($method === 'GET' && $path === '/notams') {
    requireAuth($pdo, $config['app']['jwt_secret']);
    $icao = strtoupper(trim((string) ($_GET['icao'] ?? '')));
    if ($icao === '') {
        fail('ICAO required', 400);
        return;
    }

    $number = 'A' . random_int(1000, 9999) . '/25';
    sendJson([
        'success' => true,
        'code' => 'MOCK',
        'data' => [[
            'id' => $number,
            'number' => $number,
            'type' => 'N',
            'location' => $icao,
            'start' => isoNow(),
            'end' => isoNow(86400),
            'text' => 'RWY 04L/22R CLSD DUE WIP. MAINT VEHICLES ON TWY A.',
            'source' => getApiKey($pdo, $config['api_keys'], 'icao') !== null ? 'ICAO' : 'MOCK',
        ]],
    ]);
    return;
}

if ($method === 'GET' && $path === '/openaip/airports/nearby') {
    requireAuth($pdo, $config['app']['jwt_secret']);

    $lat = parseFloatQuery($_GET['lat'] ?? null);
    $lon = parseFloatQuery($_GET['lon'] ?? null);
    $distNm = clampFloat((float) (parseFloatQuery($_GET['distNm'] ?? null) ?? 50.0), 5.0, 300.0);
    $limit = clampInt((int) (parseIntQuery($_GET['limit'] ?? null) ?? 8), 1, 25);

    if ($lat === null || $lon === null) {
        fail('Valid lat and lon query params are required', 400);
        return;
    }

    try {
        $distMeters = (int) round($distNm * 1852);
        $response = fetchOpenAipCore($pdo, $config['api_keys'], '/airports', [
            'pos' => $lat . ',' . $lon,
            'dist' => $distMeters,
            'limit' => clampInt($limit * 4, 10, 100),
            'sortBy' => 'name',
        ]);

        $items = isset($response['items']) && is_array($response['items']) ? $response['items'] : [];
        $airports = [];

        foreach ($items as $item) {
            if (!is_array($item)) {
                continue;
            }

            $center = geometryCenter(isset($item['geometry']) && is_array($item['geometry']) ? $item['geometry'] : null);
            if ($center === null) {
                continue;
            }

            $distance = haversineDistanceNm($lat, $lon, (float) $center['lat'], (float) $center['lon']);
            if ($distance > $distNm) {
                continue;
            }

            $airports[] = [
                'id' => $item['_id'] ?? null,
                'name' => $item['name'] ?? 'Unknown Airport',
                'type' => $item['type'] ?? null,
                'country' => $item['country'] ?? null,
                'latitude' => (float) $center['lat'],
                'longitude' => (float) $center['lon'],
                'distanceNm' => round($distance, 1),
                'bearingDegrees' => (int) round(bearingDegrees($lat, $lon, (float) $center['lat'], (float) $center['lon'])),
                'elevation' => $item['elevation'] ?? null,
            ];
        }

        usort($airports, static fn (array $a, array $b): int => $a['distanceNm'] <=> $b['distanceNm']);
        $airports = array_slice($airports, 0, $limit);

        sendJson([
            'success' => true,
            'data' => $airports,
            'meta' => [
                'source' => 'openaip-core',
                'cacheTtlSeconds' => 60,
            ],
        ]);
    } catch (Throwable $exception) {
        fail('Failed to fetch nearby airports from OpenAIP', 500);
    }
    return;
}

if ($method === 'GET' && $path === '/openaip/airspaces/nearby') {
    requireAuth($pdo, $config['app']['jwt_secret']);

    $lat = parseFloatQuery($_GET['lat'] ?? null);
    $lon = parseFloatQuery($_GET['lon'] ?? null);
    $distNm = clampFloat((float) (parseFloatQuery($_GET['distNm'] ?? null) ?? 80.0), 10.0, 400.0);
    $limit = clampInt((int) (parseIntQuery($_GET['limit'] ?? null) ?? 15), 1, 50);

    if ($lat === null || $lon === null) {
        fail('Valid lat and lon query params are required', 400);
        return;
    }

    try {
        $distMeters = (int) round($distNm * 1852);
        $response = fetchOpenAipCore($pdo, $config['api_keys'], '/airspaces', [
            'pos' => $lat . ',' . $lon,
            'dist' => $distMeters,
            'limit' => clampInt($limit * 3, 10, 120),
            'sortBy' => 'name',
        ]);

        $items = isset($response['items']) && is_array($response['items']) ? $response['items'] : [];
        $airspaces = [];

        foreach ($items as $item) {
            if (!is_array($item)) {
                continue;
            }

            $center = geometryCenter(isset($item['geometry']) && is_array($item['geometry']) ? $item['geometry'] : null);
            if ($center === null) {
                continue;
            }

            $distance = haversineDistanceNm($lat, $lon, (float) $center['lat'], (float) $center['lon']);
            if ($distance > $distNm) {
                continue;
            }

            $airspaces[] = [
                'id' => $item['_id'] ?? null,
                'name' => $item['name'] ?? 'Unnamed Airspace',
                'type' => $item['type'] ?? null,
                'icaoClass' => $item['icaoClass'] ?? null,
                'lowerLimit' => $item['lowerLimit'] ?? null,
                'upperLimit' => $item['upperLimit'] ?? null,
                'activity' => $item['activity'] ?? null,
                'byNotam' => (bool) ($item['byNotam'] ?? false),
                'onDemand' => (bool) ($item['onDemand'] ?? false),
                'onRequest' => (bool) ($item['onRequest'] ?? false),
                'latitude' => (float) $center['lat'],
                'longitude' => (float) $center['lon'],
                'distanceNm' => round($distance, 1),
            ];
        }

        usort($airspaces, static fn (array $a, array $b): int => $a['distanceNm'] <=> $b['distanceNm']);
        $airspaces = array_slice($airspaces, 0, $limit);

        sendJson([
            'success' => true,
            'data' => $airspaces,
            'meta' => [
                'source' => 'openaip-core',
                'cacheTtlSeconds' => 60,
            ],
        ]);
    } catch (Throwable $exception) {
        fail('Failed to fetch nearby airspaces from OpenAIP', 500);
    }
    return;
}

if ($method === 'GET' && $path === '/openaip/obstacles/nearby') {
    requireAuth($pdo, $config['app']['jwt_secret']);

    $lat = parseFloatQuery($_GET['lat'] ?? null);
    $lon = parseFloatQuery($_GET['lon'] ?? null);
    $distNm = clampFloat((float) (parseFloatQuery($_GET['distNm'] ?? null) ?? 20.0), 1.0, 150.0);
    $limit = clampInt((int) (parseIntQuery($_GET['limit'] ?? null) ?? 20), 1, 50);

    if ($lat === null || $lon === null) {
        fail('Valid lat and lon query params are required', 400);
        return;
    }

    try {
        $distMeters = (int) round($distNm * 1852);
        $response = fetchOpenAipCore($pdo, $config['api_keys'], '/obstacles', [
            'pos' => $lat . ',' . $lon,
            'dist' => $distMeters,
            'limit' => clampInt($limit * 3, 10, 120),
            'sortBy' => 'name',
        ]);

        $items = isset($response['items']) && is_array($response['items']) ? $response['items'] : [];
        $obstacles = [];

        foreach ($items as $item) {
            if (!is_array($item)) {
                continue;
            }

            $center = geometryCenter(isset($item['geometry']) && is_array($item['geometry']) ? $item['geometry'] : null);
            if ($center === null) {
                continue;
            }

            $distance = haversineDistanceNm($lat, $lon, (float) $center['lat'], (float) $center['lon']);
            if ($distance > $distNm) {
                continue;
            }

            $obstacles[] = [
                'id' => $item['_id'] ?? null,
                'name' => $item['name'] ?? 'Unnamed Obstacle',
                'type' => $item['type'] ?? null,
                'country' => $item['country'] ?? null,
                'height' => $item['height'] ?? null,
                'elevation' => $item['elevation'] ?? null,
                'latitude' => (float) $center['lat'],
                'longitude' => (float) $center['lon'],
                'distanceNm' => round($distance, 1),
            ];
        }

        usort($obstacles, static fn (array $a, array $b): int => $a['distanceNm'] <=> $b['distanceNm']);
        $obstacles = array_slice($obstacles, 0, $limit);

        sendJson([
            'success' => true,
            'data' => $obstacles,
            'meta' => [
                'source' => 'openaip-core',
                'cacheTtlSeconds' => 60,
            ],
        ]);
    } catch (Throwable $exception) {
        fail('Failed to fetch nearby obstacles from OpenAIP', 500);
    }
    return;
}

if ($method === 'GET' && $path === '/openaip/smart-alerts') {
    requireAuth($pdo, $config['app']['jwt_secret']);

    $lat = parseFloatQuery($_GET['lat'] ?? null);
    $lon = parseFloatQuery($_GET['lon'] ?? null);

    if ($lat === null || $lon === null) {
        fail('Valid lat and lon query params are required', 400);
        return;
    }

    try {
        $airportResponse = fetchOpenAipCore($pdo, $config['api_keys'], '/airports', [
            'pos' => $lat . ',' . $lon,
            'dist' => (int) round(40 * 1852),
            'limit' => 20,
        ]);
        $airspaceResponse = fetchOpenAipCore($pdo, $config['api_keys'], '/airspaces', [
            'pos' => $lat . ',' . $lon,
            'dist' => (int) round(80 * 1852),
            'limit' => 30,
        ]);
        $obstacleResponse = fetchOpenAipCore($pdo, $config['api_keys'], '/obstacles', [
            'pos' => $lat . ',' . $lon,
            'dist' => (int) round(20 * 1852),
            'limit' => 20,
        ]);

        $airports = [];
        foreach (($airportResponse['items'] ?? []) as $item) {
            if (!is_array($item)) continue;
            $center = geometryCenter(isset($item['geometry']) && is_array($item['geometry']) ? $item['geometry'] : null);
            if ($center === null) continue;
            $distance = haversineDistanceNm($lat, $lon, (float) $center['lat'], (float) $center['lon']);
            $airports[] = [
                'id' => $item['_id'] ?? null,
                'name' => $item['name'] ?? 'Unknown Airport',
                'distanceNm' => round($distance, 1),
                'bearingDegrees' => (int) round(bearingDegrees($lat, $lon, (float) $center['lat'], (float) $center['lon'])),
            ];
        }
        usort($airports, static fn (array $a, array $b): int => $a['distanceNm'] <=> $b['distanceNm']);

        $airspaces = [];
        foreach (($airspaceResponse['items'] ?? []) as $item) {
            if (!is_array($item)) continue;
            $center = geometryCenter(isset($item['geometry']) && is_array($item['geometry']) ? $item['geometry'] : null);
            if ($center === null) continue;
            $distance = haversineDistanceNm($lat, $lon, (float) $center['lat'], (float) $center['lon']);
            $airspaces[] = [
                'id' => $item['_id'] ?? null,
                'name' => $item['name'] ?? 'Unnamed Airspace',
                'icaoClass' => $item['icaoClass'] ?? null,
                'lowerLimit' => $item['lowerLimit'] ?? null,
                'upperLimit' => $item['upperLimit'] ?? null,
                'distanceNm' => round($distance, 1),
            ];
        }
        usort($airspaces, static fn (array $a, array $b): int => $a['distanceNm'] <=> $b['distanceNm']);

        $obstacles = [];
        foreach (($obstacleResponse['items'] ?? []) as $item) {
            if (!is_array($item)) continue;
            $center = geometryCenter(isset($item['geometry']) && is_array($item['geometry']) ? $item['geometry'] : null);
            if ($center === null) continue;
            $distance = haversineDistanceNm($lat, $lon, (float) $center['lat'], (float) $center['lon']);
            $obstacles[] = [
                'id' => $item['_id'] ?? null,
                'name' => $item['name'] ?? 'Obstacle',
                'distanceNm' => round($distance, 1),
                'height' => $item['height'] ?? null,
            ];
        }
        usort($obstacles, static fn (array $a, array $b): int => $a['distanceNm'] <=> $b['distanceNm']);

        $alerts = [];

        $nearestAirport = $airports[0] ?? null;
        if ($nearestAirport !== null) {
            $alerts[] = [
                'id' => 'nearest-airport-' . ($nearestAirport['id'] ?? 'unknown'),
                'level' => ($nearestAirport['distanceNm'] <= 10.0) ? 'info' : 'notice',
                'title' => 'Nearest Airport',
                'message' => $nearestAirport['name'] . ' is ' . $nearestAirport['distanceNm'] . 'nm away on bearing ' . $nearestAirport['bearingDegrees'] . 'deg.',
                'source' => 'openAIP',
            ];
        }

        foreach (array_slice(array_filter($airspaces, static fn (array $a): bool => $a['distanceNm'] <= 15.0), 0, 3) as $airspace) {
            $alerts[] = [
                'id' => 'airspace-' . ($airspace['id'] ?? uniqid('', true)),
                'level' => 'warning',
                'title' => 'Nearby Airspace',
                'message' => $airspace['name'] . ' (' . $airspace['distanceNm'] . 'nm). Check limits before route entry.',
                'source' => 'openAIP',
            ];
        }

        foreach (array_slice(array_filter($obstacles, static fn (array $o): bool => $o['distanceNm'] <= 5.0), 0, 3) as $obstacle) {
            $alerts[] = [
                'id' => 'obstacle-' . ($obstacle['id'] ?? uniqid('', true)),
                'level' => 'caution',
                'title' => 'Obstacle Proximity',
                'message' => $obstacle['name'] . ' detected within ' . $obstacle['distanceNm'] . 'nm.',
                'source' => 'openAIP',
            ];
        }

        if (count($alerts) === 0) {
            $alerts[] = [
                'id' => 'no-alerts',
                'level' => 'info',
                'title' => 'No Immediate Alerts',
                'message' => 'No nearby airspace or obstacle warnings detected for current position.',
                'source' => 'openAIP',
            ];
        }

        sendJson([
            'success' => true,
            'data' => $alerts,
            'meta' => [
                'nearestAirport' => $nearestAirport,
                'nearbyAirspaces' => array_slice($airspaces, 0, 5),
                'nearbyObstacles' => array_slice($obstacles, 0, 5),
                'source' => 'openaip-core',
                'cacheTtlSeconds' => 60,
            ],
        ]);
    } catch (Throwable $exception) {
        fail('Failed to generate OpenAIP smart alerts', 500);
    }
    return;
}

// 3. PIREPs
if ($method === 'GET' && $path === '/pireps') {
    requireAuth($pdo, $config['app']['jwt_secret']);
    try {
        $stmt = $pdo->query(
            'SELECT p.*, u.username AS submitted_by_username
             FROM pireps p
             LEFT JOIN users u ON p.submitted_by = u.id
             ORDER BY p.created_at DESC'
        );
        $rows = $stmt === false ? [] : $stmt->fetchAll();

        $formatted = array_map(static function (array $pirep): array {
            return [
                'id' => (int) $pirep['id'],
                'icao_code' => $pirep['icao_code'],
                'aircraft_type' => $pirep['aircraft_type'],
                'flight_level' => $pirep['flight_level'],
                'latitude' => $pirep['latitude'] !== null ? (float) $pirep['latitude'] : null,
                'longitude' => $pirep['longitude'] !== null ? (float) $pirep['longitude'] : null,
                'weather_conditions' => $pirep['weather_conditions'],
                'turbulence' => $pirep['turbulence'],
                'icing' => $pirep['icing'],
                'remarks' => $pirep['remarks'],
                'submitted_by' => $pirep['submitted_by'],
                'created_at' => $pirep['created_at'],
                'submittedByUsername' => $pirep['submitted_by_username'] ?? null,
            ];
        }, $rows);

        sendJson(['success' => true, 'data' => $formatted]);
    } catch (Throwable $exception) {
        fail('Failed to fetch PIREPs', 500);
    }
    return;
}

if ($method === 'POST' && $path === '/pireps') {
    $authUser = requireAuth($pdo, $config['app']['jwt_secret']);
    $payload = jsonBody();
    $position = is_array($payload['aircraft_position'] ?? null) ? $payload['aircraft_position'] : [];

    try {
        $insert = $pdo->prepare(
            'INSERT INTO pireps
             (icao_code, aircraft_type, flight_level, latitude, longitude, weather_conditions, turbulence, icing, remarks, submitted_by)
             VALUES (:icao_code, :aircraft_type, :flight_level, :latitude, :longitude, :weather_conditions, :turbulence, :icing, :remarks, :submitted_by)'
        );
        $insert->execute([
            'icao_code' => strtoupper((string) ($payload['icao_code'] ?? '')),
            'aircraft_type' => (string) ($payload['aircraft_type'] ?? ''),
            'flight_level' => (string) ($payload['flight_level'] ?? ''),
            'latitude' => isset($position['latitude']) ? (float) $position['latitude'] : null,
            'longitude' => isset($position['longitude']) ? (float) $position['longitude'] : null,
            'weather_conditions' => (string) ($payload['weather_conditions'] ?? ''),
            'turbulence' => (string) ($payload['turbulence'] ?? ''),
            'icing' => (string) ($payload['icing'] ?? ''),
            'remarks' => (string) ($payload['remarks'] ?? ''),
            'submitted_by' => $authUser['id'],
        ]);

        $id = (int) $pdo->lastInsertId();
        $select = $pdo->prepare(
            'SELECT p.*, u.username AS submitted_by_username
             FROM pireps p
             LEFT JOIN users u ON p.submitted_by = u.id
             WHERE p.id = :id
             LIMIT 1'
        );
        $select->execute(['id' => $id]);
        $created = $select->fetch();

        if (!$created) {
            fail('Failed to submit PIREP', 500);
            return;
        }

        sendJson([
            'success' => true,
            'data' => [
                'id' => (int) $created['id'],
                'icao_code' => $created['icao_code'],
                'aircraft_type' => $created['aircraft_type'],
                'flight_level' => $created['flight_level'],
                'latitude' => $created['latitude'] !== null ? (float) $created['latitude'] : null,
                'longitude' => $created['longitude'] !== null ? (float) $created['longitude'] : null,
                'weather_conditions' => $created['weather_conditions'],
                'turbulence' => $created['turbulence'],
                'icing' => $created['icing'],
                'remarks' => $created['remarks'],
                'submitted_by' => $created['submitted_by'],
                'created_at' => $created['created_at'],
                'submittedByUsername' => $created['submitted_by_username'] ?? null,
            ],
        ]);
    } catch (Throwable $exception) {
        fail('Failed to submit PIREP', 500);
    }
    return;
}

// 4. Airports
if ($method === 'GET' && $path === '/airports') {
    requireAuth($pdo, $config['app']['jwt_secret']);
    $icao = strtoupper(trim((string) ($_GET['icao'] ?? '')));
    if ($icao === '') {
        fail('ICAO required', 400);
        return;
    }

    $stmt = $pdo->prepare('SELECT * FROM airports WHERE icao = :icao LIMIT 1');
    $stmt->execute(['icao' => $icao]);
    $airport = $stmt->fetch();

    if (!$airport) {
        fail('Airport not found', 404);
        return;
    }

    foreach (['runways', 'frequencies', 'procedures'] as $field) {
        $airport[$field] = safeJsonDecode($airport[$field] ?? null, []);
    }

    sendJson(['success' => true, 'data' => $airport]);
    return;
}

if ($method === 'GET' && $path === '/airports/all') {
    requireAuth($pdo, $config['app']['jwt_secret']);
    try {
        $hasBounds = isset($_GET['minLat'], $_GET['maxLat'], $_GET['minLng'], $_GET['maxLng']);
        if ($hasBounds) {
            $stmt = $pdo->prepare(
                'SELECT * FROM airports
                 WHERE latitude BETWEEN :minLat AND :maxLat
                   AND longitude BETWEEN :minLng AND :maxLng'
            );
            $stmt->execute([
                'minLat' => (float) $_GET['minLat'],
                'maxLat' => (float) $_GET['maxLat'],
                'minLng' => (float) $_GET['minLng'],
                'maxLng' => (float) $_GET['maxLng'],
            ]);
        } else {
            $stmt = $pdo->query('SELECT * FROM airports');
        }

        $airports = $stmt === false ? [] : $stmt->fetchAll();
        foreach ($airports as &$airport) {
            foreach (['runways', 'frequencies', 'procedures'] as $field) {
                $airport[$field] = safeJsonDecode($airport[$field] ?? null, []);
            }
        }
        unset($airport);

        sendJson(['success' => true, 'data' => $airports]);
    } catch (Throwable $exception) {
        fail('Failed to fetch airport data', 500);
    }
    return;
}

// 5. Flight Plans
if ($method === 'POST' && $path === '/flight-plans') {
    $authUser = requireAuth($pdo, $config['app']['jwt_secret']);
    $payload = jsonBody();

    try {
        $stmt = $pdo->prepare(
            'INSERT INTO flight_plans (user_id, departure_icao, destination_icao, aircraft_type, planned_departure, flight_rules, route)
             VALUES (:user_id, :departure_icao, :destination_icao, :aircraft_type, :planned_departure, :flight_rules, :route)'
        );
        $stmt->execute([
            'user_id' => $authUser['id'],
            'departure_icao' => strtoupper((string) ($payload['departure_icao'] ?? '')),
            'destination_icao' => strtoupper((string) ($payload['destination_icao'] ?? '')),
            'aircraft_type' => (string) ($payload['aircraft_type'] ?? ''),
            'planned_departure' => (string) ($payload['planned_departure'] ?? null),
            'flight_rules' => (string) ($payload['flight_rules'] ?? 'VFR'),
            'route' => json_encode($payload['route'] ?? [], JSON_UNESCAPED_SLASHES),
        ]);

        sendJson(['success' => true, 'data' => ['id' => (int) $pdo->lastInsertId()]]);
    } catch (Throwable $exception) {
        fail('Failed to save flight plan', 500);
    }
    return;
}

if ($method === 'GET' && $path === '/flight-plans') {
    $authUser = requireAuth($pdo, $config['app']['jwt_secret']);
    $stmt = $pdo->prepare('SELECT * FROM flight_plans WHERE user_id = :user_id ORDER BY created_at DESC');
    $stmt->execute(['user_id' => $authUser['id']]);
    $plans = $stmt->fetchAll();

    $formatted = array_map(static function (array $plan): array {
        $plan['route'] = safeJsonDecode($plan['route'] ?? null, []);
        return $plan;
    }, $plans);

    sendJson(['success' => true, 'data' => $formatted]);
    return;
}

// 6. Admin Routes
if ($method === 'GET' && $path === '/admin/users') {
    $authUser = requireAuth($pdo, $config['app']['jwt_secret']);
    requireAdmin($authUser);

    $stmt = $pdo->query('SELECT id, email, username, role, created_at FROM users ORDER BY created_at DESC');
    $users = $stmt === false ? [] : $stmt->fetchAll();
    $formatted = array_map(static function (array $user): array {
        return [
            'id' => $user['id'],
            'email' => $user['email'],
            'username' => $user['username'] ?? explode('@', (string) $user['email'])[0] ?? 'Pilot',
            'role' => $user['role'] ?? 'user',
            'createdAt' => $user['created_at'] ?? null,
        ];
    }, $users);

    sendJson(['success' => true, 'data' => $formatted]);
    return;
}

if ($method === 'DELETE' && preg_match('#^/admin/users/([^/]+)$#', $path, $matches) === 1) {
    $authUser = requireAuth($pdo, $config['app']['jwt_secret']);
    requireAdmin($authUser);

    $targetUserId = (string) $matches[1];
    if ($targetUserId === (string) $authUser['id']) {
        fail('You cannot delete your own account', 400);
        return;
    }

    $targetStmt = $pdo->prepare('SELECT id, role FROM users WHERE id = :id LIMIT 1');
    $targetStmt->execute(['id' => $targetUserId]);
    $targetUser = $targetStmt->fetch();
    if (!$targetUser) {
        fail('User not found', 404);
        return;
    }

    if (($targetUser['role'] ?? 'user') === 'admin') {
        $countStmt = $pdo->query("SELECT COUNT(*) AS count FROM users WHERE role = 'admin'");
        $adminCount = (int) (($countStmt === false ? [] : $countStmt->fetch())['count'] ?? 0);
        if ($adminCount <= 1) {
            fail('Cannot delete the last administrator', 400);
            return;
        }
    }

    $stmt = $pdo->prepare('DELETE FROM users WHERE id = :id');
    $stmt->execute(['id' => $targetUserId]);

    sendJson(['success' => true]);
    return;
}

if ($method === 'PUT' && preg_match('#^/admin/users/([^/]+)/role$#', $path, $matches) === 1) {
    $authUser = requireAuth($pdo, $config['app']['jwt_secret']);
    requireAdmin($authUser);

    $payload = jsonBody();
    $role = (string) ($payload['role'] ?? '');
    if (!in_array($role, ['user', 'admin'], true)) {
        fail('Invalid role', 400);
        return;
    }

    $targetUserId = (string) $matches[1];
    if ($targetUserId === (string) $authUser['id']) {
        fail('You cannot change your own role from the admin console', 400);
        return;
    }

    $targetStmt = $pdo->prepare('SELECT id, role FROM users WHERE id = :id LIMIT 1');
    $targetStmt->execute(['id' => $targetUserId]);
    $targetUser = $targetStmt->fetch();
    if (!$targetUser) {
        fail('User not found', 404);
        return;
    }

    $currentRole = (string) ($targetUser['role'] ?? 'user');
    if ($currentRole === 'admin' && $role !== 'admin') {
        $countStmt = $pdo->query("SELECT COUNT(*) AS count FROM users WHERE role = 'admin'");
        $adminCount = (int) (($countStmt === false ? [] : $countStmt->fetch())['count'] ?? 0);
        if ($adminCount <= 1) {
            fail('Cannot demote the last administrator', 400);
            return;
        }
    }

    $stmt = $pdo->prepare('UPDATE users SET role = :role WHERE id = :id');
    $stmt->execute(['role' => $role, 'id' => $targetUserId]);

    sendJson(['success' => true]);
    return;
}

if ($method === 'GET' && $path === '/admin/platform/summary') {
    $authUser = requireAuth($pdo, $config['app']['jwt_secret']);
    requireAdmin($authUser);

    $counts = [
        'users' => 0,
        'admins' => 0,
        'airports' => 0,
        'pireps' => 0,
        'flightPlans' => 0,
    ];

    try {
        $queries = [
            'users' => 'SELECT COUNT(*) AS count FROM users',
            'admins' => "SELECT COUNT(*) AS count FROM users WHERE role = 'admin'",
            'airports' => 'SELECT COUNT(*) AS count FROM airports',
            'pireps' => 'SELECT COUNT(*) AS count FROM pireps',
            'flightPlans' => 'SELECT COUNT(*) AS count FROM flight_plans',
        ];

        foreach ($queries as $key => $sql) {
            $stmt = $pdo->query($sql);
            $row = $stmt === false ? [] : $stmt->fetch();
            $counts[$key] = (int) ($row['count'] ?? 0);
        }
    } catch (Throwable $exception) {
        // Return default zero counts if query fails.
    }

    $keys = getApiKeys($pdo, $config['api_keys']);
    $apiKeyStatus = [
        'checkwx' => !empty($keys['checkwx']) ? 'configured' : 'missing',
        'icao' => !empty($keys['icao']) ? 'configured' : 'missing',
        'openweather' => !empty($keys['openweather']) ? 'configured' : 'missing',
        'windy' => !empty($keys['windy']) ? 'configured' : 'missing',
        'avwx' => !empty($keys['avwx']) ? 'configured' : 'missing',
        'openaip' => !empty($keys['openaip']) ? 'configured' : 'missing',
    ];

    $databaseHealthy = false;
    try {
        $healthStmt = $pdo->query('SELECT 1');
        $databaseHealthy = $healthStmt !== false;
    } catch (Throwable $exception) {
        $databaseHealthy = false;
    }

    $recentActivity = [];
    try {
        $usersStmt = $pdo->query('SELECT id, email, username, created_at FROM users ORDER BY created_at DESC LIMIT 12');
        $userRows = $usersStmt === false ? [] : $usersStmt->fetchAll();
        foreach ($userRows as $row) {
            $username = (string) ($row['username'] ?? '');
            $email = (string) ($row['email'] ?? '');
            $recentActivity[] = [
                'id' => 'user-' . (string) ($row['id'] ?? uniqid('', true)),
                'type' => 'user_registered',
                'timestamp' => (string) ($row['created_at'] ?? gmdate('Y-m-d H:i:s')),
                'level' => 'info',
                'message' => 'User registration: ' . ($username !== '' ? $username : $email),
            ];
        }

        $pirepStmt = $pdo->query(
            'SELECT p.id, p.icao_code, p.created_at, COALESCE(u.username, u.email, "Unknown") AS reporter
             FROM pireps p
             LEFT JOIN users u ON p.submitted_by = u.id
             ORDER BY p.created_at DESC
             LIMIT 12'
        );
        $pirepRows = $pirepStmt === false ? [] : $pirepStmt->fetchAll();
        foreach ($pirepRows as $row) {
            $recentActivity[] = [
                'id' => 'pirep-' . (string) ($row['id'] ?? uniqid('', true)),
                'type' => 'pirep_submitted',
                'timestamp' => (string) ($row['created_at'] ?? gmdate('Y-m-d H:i:s')),
                'level' => 'notice',
                'message' => 'PIREP submitted for ' . (string) ($row['icao_code'] ?? 'N/A') . ' by ' . (string) ($row['reporter'] ?? 'Unknown'),
            ];
        }

        $planStmt = $pdo->query(
            'SELECT fp.id, fp.departure_icao, fp.destination_icao, fp.created_at, COALESCE(u.username, u.email, "Unknown") AS planner
             FROM flight_plans fp
             LEFT JOIN users u ON fp.user_id = u.id
             ORDER BY fp.created_at DESC
             LIMIT 12'
        );
        $planRows = $planStmt === false ? [] : $planStmt->fetchAll();
        foreach ($planRows as $row) {
            $recentActivity[] = [
                'id' => 'plan-' . (string) ($row['id'] ?? uniqid('', true)),
                'type' => 'flight_plan_saved',
                'timestamp' => (string) ($row['created_at'] ?? gmdate('Y-m-d H:i:s')),
                'level' => 'info',
                'message' => 'Flight plan saved: '
                    . (string) ($row['departure_icao'] ?? 'N/A')
                    . ' -> '
                    . (string) ($row['destination_icao'] ?? 'N/A')
                    . ' by '
                    . (string) ($row['planner'] ?? 'Unknown'),
            ];
        }

        $keyStmt = $pdo->query('SELECT name, updated_at FROM api_keys ORDER BY updated_at DESC LIMIT 12');
        $keyRows = $keyStmt === false ? [] : $keyStmt->fetchAll();
        foreach ($keyRows as $row) {
            $recentActivity[] = [
                'id' => 'key-' . (string) ($row['name'] ?? uniqid('', true)) . '-' . (string) ($row['updated_at'] ?? ''),
                'type' => 'api_key_updated',
                'timestamp' => (string) ($row['updated_at'] ?? gmdate('Y-m-d H:i:s')),
                'level' => 'warning',
                'message' => 'API key updated: ' . (string) ($row['name'] ?? 'unknown'),
            ];
        }
    } catch (Throwable $exception) {
        // Keep activity empty when queries fail.
    }

    usort($recentActivity, static function (array $a, array $b): int {
        $left = strtotime((string) ($a['timestamp'] ?? '')) ?: 0;
        $right = strtotime((string) ($b['timestamp'] ?? '')) ?: 0;
        return $right <=> $left;
    });

    sendJson([
        'success' => true,
        'data' => [
            'counts' => $counts,
            'apiKeyStatus' => $apiKeyStatus,
            'databaseHealthy' => $databaseHealthy,
            'platformSettings' => readPlatformSettings($pdo),
            'recentActivity' => array_slice($recentActivity, 0, 30),
            'generatedAt' => gmdate('c'),
        ],
    ]);
    return;
}

if ($method === 'GET' && $path === '/admin/platform/settings') {
    $authUser = requireAuth($pdo, $config['app']['jwt_secret']);
    requireAdmin($authUser);

    sendJson([
        'success' => true,
        'data' => readPlatformSettings($pdo),
    ]);
    return;
}

if ($method === 'PUT' && $path === '/admin/platform/settings') {
    $authUser = requireAuth($pdo, $config['app']['jwt_secret']);
    requireAdmin($authUser);

    $payload = jsonBody();
    if (!is_array($payload)) {
        $payload = [];
    }

    try {
        $saved = savePlatformSettings($pdo, $payload);
        sendJson([
            'success' => true,
            'data' => $saved,
        ]);
    } catch (Throwable $exception) {
        fail('Failed to update platform settings', 500);
    }
    return;
}

if ($method === 'GET' && $path === '/admin/airports') {
    $authUser = requireAuth($pdo, $config['app']['jwt_secret']);
    requireAdmin($authUser);

    $query = trim((string) ($_GET['q'] ?? ''));
    $limit = clampInt((int) (parseIntQuery($_GET['limit'] ?? null) ?? 50), 1, 200);
    $offset = max(0, (int) (parseIntQuery($_GET['offset'] ?? null) ?? 0));

    try {
        $total = 0;
        if ($query !== '') {
            $like = '%' . $query . '%';
            $countStmt = $pdo->prepare('SELECT COUNT(*) AS count FROM airports WHERE icao LIKE :like OR name LIKE :like');
            $countStmt->execute(['like' => $like]);
            $total = (int) (($countStmt->fetch())['count'] ?? 0);

            $stmt = $pdo->prepare(
                'SELECT * FROM airports
                 WHERE icao LIKE :like OR name LIKE :like
                 ORDER BY icao ASC
                 LIMIT :limit OFFSET :offset'
            );
            $stmt->bindValue(':like', $like, PDO::PARAM_STR);
            $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
            $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
            $stmt->execute();
        } else {
            $countStmt = $pdo->query('SELECT COUNT(*) AS count FROM airports');
            $total = (int) (($countStmt === false ? [] : $countStmt->fetch())['count'] ?? 0);

            $stmt = $pdo->prepare('SELECT * FROM airports ORDER BY icao ASC LIMIT :limit OFFSET :offset');
            $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
            $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
            $stmt->execute();
        }

        $rows = $stmt->fetchAll();
        $formatted = array_map(static function (array $airport): array {
            $runways = safeJsonDecode($airport['runways'] ?? null, []);
            $frequencies = safeJsonDecode($airport['frequencies'] ?? null, []);
            $procedures = safeJsonDecode($airport['procedures'] ?? null, []);

            return [
                'icao' => $airport['icao'],
                'name' => $airport['name'],
                'elevation' => $airport['elevation'] !== null ? (int) $airport['elevation'] : null,
                'latitude' => $airport['latitude'] !== null ? (float) $airport['latitude'] : null,
                'longitude' => $airport['longitude'] !== null ? (float) $airport['longitude'] : null,
                'runways' => is_array($runways) ? $runways : [],
                'frequencies' => is_array($frequencies) ? $frequencies : [],
                'procedures' => is_array($procedures) ? $procedures : [],
                'runwayCount' => is_array($runways) ? count($runways) : 0,
                'frequencyCount' => is_array($frequencies) ? count($frequencies) : 0,
                'procedureCount' => is_array($procedures) ? count($procedures) : 0,
            ];
        }, $rows);

        sendJson([
            'success' => true,
            'data' => $formatted,
            'meta' => [
                'total' => $total,
                'limit' => $limit,
                'offset' => $offset,
                'query' => $query,
            ],
        ]);
    } catch (Throwable $exception) {
        fail('Failed to fetch admin airport list', 500);
    }
    return;
}

if ($method === 'PUT' && preg_match('#^/admin/airports/([A-Za-z0-9]{3,8})$#', $path, $matches) === 1) {
    $authUser = requireAuth($pdo, $config['app']['jwt_secret']);
    requireAdmin($authUser);

    $icao = strtoupper((string) $matches[1]);
    $payload = jsonBody();

    $existingStmt = $pdo->prepare('SELECT * FROM airports WHERE icao = :icao LIMIT 1');
    $existingStmt->execute(['icao' => $icao]);
    $existing = $existingStmt->fetch();

    $providedName = trim((string) ($payload['name'] ?? ''));
    if (!$existing && $providedName === '') {
        fail('Airport name is required for new airport records', 400);
        return;
    }

    foreach (['runways', 'frequencies', 'procedures'] as $listField) {
        if (array_key_exists($listField, $payload) && !is_array($payload[$listField])) {
            fail($listField . ' must be an array', 400);
            return;
        }
    }

    $runways = array_key_exists('runways', $payload)
        ? $payload['runways']
        : safeJsonDecode($existing['runways'] ?? null, []);
    $frequencies = array_key_exists('frequencies', $payload)
        ? $payload['frequencies']
        : safeJsonDecode($existing['frequencies'] ?? null, []);
    $procedures = array_key_exists('procedures', $payload)
        ? $payload['procedures']
        : safeJsonDecode($existing['procedures'] ?? null, []);

    $name = $providedName !== '' ? $providedName : (string) ($existing['name'] ?? '');
    $elevation = array_key_exists('elevation', $payload)
        ? ($payload['elevation'] === null || $payload['elevation'] === '' ? null : (int) $payload['elevation'])
        : (isset($existing['elevation']) ? (int) $existing['elevation'] : null);
    $latitude = array_key_exists('latitude', $payload)
        ? ($payload['latitude'] === null || $payload['latitude'] === '' ? null : (float) $payload['latitude'])
        : (isset($existing['latitude']) ? (float) $existing['latitude'] : null);
    $longitude = array_key_exists('longitude', $payload)
        ? ($payload['longitude'] === null || $payload['longitude'] === '' ? null : (float) $payload['longitude'])
        : (isset($existing['longitude']) ? (float) $existing['longitude'] : null);

    try {
        $upsert = $pdo->prepare(
            'INSERT INTO airports (icao, name, elevation, latitude, longitude, runways, frequencies, procedures)
             VALUES (:icao, :name, :elevation, :latitude, :longitude, :runways, :frequencies, :procedures)
             ON DUPLICATE KEY UPDATE
               name = VALUES(name),
               elevation = VALUES(elevation),
               latitude = VALUES(latitude),
               longitude = VALUES(longitude),
               runways = VALUES(runways),
               frequencies = VALUES(frequencies),
               procedures = VALUES(procedures)'
        );
        $upsert->execute([
            'icao' => $icao,
            'name' => $name,
            'elevation' => $elevation,
            'latitude' => $latitude,
            'longitude' => $longitude,
            'runways' => json_encode($runways ?? [], JSON_UNESCAPED_SLASHES),
            'frequencies' => json_encode($frequencies ?? [], JSON_UNESCAPED_SLASHES),
            'procedures' => json_encode($procedures ?? [], JSON_UNESCAPED_SLASHES),
        ]);

        $existingStmt->execute(['icao' => $icao]);
        $saved = $existingStmt->fetch();
        if (!$saved) {
            fail('Failed to persist airport record', 500);
            return;
        }

        $saved['runways'] = safeJsonDecode($saved['runways'] ?? null, []);
        $saved['frequencies'] = safeJsonDecode($saved['frequencies'] ?? null, []);
        $saved['procedures'] = safeJsonDecode($saved['procedures'] ?? null, []);

        sendJson([
            'success' => true,
            'data' => $saved,
        ]);
    } catch (Throwable $exception) {
        fail('Failed to save airport record', 500);
    }
    return;
}

if ($method === 'DELETE' && preg_match('#^/admin/airports/([A-Za-z0-9]{3,8})$#', $path, $matches) === 1) {
    $authUser = requireAuth($pdo, $config['app']['jwt_secret']);
    requireAdmin($authUser);

    $icao = strtoupper((string) $matches[1]);
    $stmt = $pdo->prepare('DELETE FROM airports WHERE icao = :icao');
    $stmt->execute(['icao' => $icao]);

    if ($stmt->rowCount() <= 0) {
        fail('Airport not found', 404);
        return;
    }

    sendJson(['success' => true]);
    return;
}

if ($method === 'GET' && $path === '/admin/api-keys') {
    $authUser = requireAuth($pdo, $config['app']['jwt_secret']);
    requireAdmin($authUser);

    sendJson([
        'success' => true,
        'data' => getApiKeys($pdo, $config['api_keys']),
    ]);
    return;
}

if ($method === 'PUT' && $path === '/admin/api-keys') {
    $authUser = requireAuth($pdo, $config['app']['jwt_secret']);
    requireAdmin($authUser);

    $updates = jsonBody();
    if (!is_array($updates)) {
        $updates = [];
    }

    $stmt = $pdo->prepare(
        'INSERT INTO api_keys (name, value, updated_at)
         VALUES (:name, :value, NOW())
         ON DUPLICATE KEY UPDATE value = VALUES(value), updated_at = NOW()'
    );

    foreach ($updates as $name => $value) {
        $key = trim((string) $name);
        if ($key === '') {
            continue;
        }

        $stmt->execute([
            'name' => $key,
            'value' => normalizeApiKeyValue($value),
        ]);
    }

    sendJson([
        'success' => true,
        'data' => getApiKeys($pdo, $config['api_keys'], true),
    ]);
    return;
}

if ($method === 'GET' && $path === '/public/map-keys') {
    $keys = getApiKeys($pdo, $config['api_keys']);

    sendJson([
        'success' => true,
        'data' => [
            'openweather' => $keys['openweather'] ?? null,
            'openaip' => $keys['openaip'] ?? null,
            'windy' => $keys['windy'] ?? null,
        ],
    ]);
    return;
}

if ($method === 'POST' && $path === '/admin/load-airports') {
    $authUser = requireAuth($pdo, $config['app']['jwt_secret']);
    requireAdmin($authUser);

    try {
        $candidates = [
            dirname(__DIR__) . '/data/airports.json',
            dirname(__DIR__, 2) . '/data/airports.json',
            dirname(__DIR__, 2) . '/airports.json',
        ];

        $dataPath = null;
        foreach ($candidates as $candidate) {
            if (is_file($candidate)) {
                $dataPath = $candidate;
                break;
            }
        }

        if ($dataPath === null) {
            throw new RuntimeException('airports.json not found');
        }

        $raw = file_get_contents($dataPath);
        if ($raw === false) {
            throw new RuntimeException('Failed to read airport data file');
        }

        $airports = json_decode($raw, true, 512, JSON_THROW_ON_ERROR);
        if (!is_array($airports)) {
            throw new RuntimeException('Invalid airport data file');
        }

        $insert = $pdo->prepare(
            'INSERT INTO airports (icao, name, elevation, latitude, longitude, runways, frequencies, procedures)
             VALUES (:icao, :name, :elevation, :latitude, :longitude, :runways, :frequencies, :procedures)
             ON DUPLICATE KEY UPDATE
               name = VALUES(name),
               elevation = VALUES(elevation),
               latitude = VALUES(latitude),
               longitude = VALUES(longitude),
               runways = VALUES(runways),
               frequencies = VALUES(frequencies),
               procedures = VALUES(procedures)'
        );

        $pdo->beginTransaction();
        $count = 0;
        foreach ($airports as $airport) {
            if (!is_array($airport) || empty($airport['icao']) || empty($airport['name'])) {
                continue;
            }

            $insert->execute([
                'icao' => strtoupper((string) $airport['icao']),
                'name' => (string) $airport['name'],
                'elevation' => isset($airport['elevation']) ? (int) $airport['elevation'] : null,
                'latitude' => isset($airport['latitude']) ? (float) $airport['latitude'] : null,
                'longitude' => isset($airport['longitude']) ? (float) $airport['longitude'] : null,
                'runways' => json_encode($airport['runways'] ?? [], JSON_UNESCAPED_SLASHES),
                'frequencies' => json_encode($airport['frequencies'] ?? [], JSON_UNESCAPED_SLASHES),
                'procedures' => json_encode($airport['procedures'] ?? [], JSON_UNESCAPED_SLASHES),
            ]);
            $count++;
        }
        $pdo->commit();

        sendJson(['success' => true, 'count' => $count]);
    } catch (Throwable $exception) {
        if ($pdo->inTransaction()) {
            $pdo->rollBack();
        }
        fail('Failed to load airport data', 500);
    }
    return;
}

// 7. Health
if ($method === 'GET' && $path === '/health') {
    $keys = getApiKeys($pdo, $config['api_keys']);
    $settings = readPlatformSettings($pdo);

    sendJson([
        'status' => 'ok',
        'mode' => 'MYSQL',
        'version' => '4.1.0',
        'apiKeys' => [
            'checkwx' => !empty($keys['checkwx']) ? 'configured' : 'missing',
            'icao' => !empty($keys['icao']) ? 'configured' : 'missing',
            'openweather' => !empty($keys['openweather']) ? 'configured' : 'missing',
            'windy' => !empty($keys['windy']) ? 'configured' : 'missing',
            'avwx' => !empty($keys['avwx']) ? 'configured' : 'missing',
            'openaip' => !empty($keys['openaip']) ? 'configured' : 'missing',
        ],
        'platformSettings' => [
            'maintenanceMode' => (bool) ($settings['maintenanceMode'] ?? false),
            'allowRegistrations' => (bool) ($settings['allowRegistrations'] ?? true),
            'defaultAirport' => (string) ($settings['defaultAirport'] ?? 'KJFK'),
        ],
    ]);
    return;
}

fail('Route not found', 404);
