<?php
declare(strict_types=1);

function getAuthorizationHeader(): ?string
{
    if (isset($_SERVER['HTTP_AUTHORIZATION']) && $_SERVER['HTTP_AUTHORIZATION'] !== '') {
        return trim((string) $_SERVER['HTTP_AUTHORIZATION']);
    }

    if (function_exists('apache_request_headers')) {
        $headers = apache_request_headers();
        foreach ($headers as $key => $value) {
            if (strtolower((string) $key) === 'authorization') {
                return trim((string) $value);
            }
        }
    }

    return null;
}

function getBearerToken(): ?string
{
    $header = getAuthorizationHeader();
    if ($header === null) {
        return null;
    }

    if (!preg_match('/^Bearer\s+(.+)$/i', $header, $matches)) {
        return null;
    }

    return trim($matches[1]);
}

function requireAuth(PDO $pdo, string $jwtSecret): array
{
    $token = getBearerToken();
    if ($token === null) {
        fail('No authorization header', 401);
        exit;
    }

    $decoded = jwtDecode($token, $jwtSecret);
    if (!is_array($decoded) || empty($decoded['id'])) {
        fail('Invalid or expired token', 401);
        exit;
    }

    $stmt = $pdo->prepare('SELECT id, email, role, username FROM users WHERE id = :id LIMIT 1');
    $stmt->execute(['id' => $decoded['id']]);
    $user = $stmt->fetch();

    if (!$user) {
        fail('User not found or token invalid', 401);
        exit;
    }

    return $user;
}

function requireAdmin(array $user): void
{
    if (($user['role'] ?? 'user') !== 'admin') {
        fail('Forbidden: Admin access required', 403);
        exit;
    }
}

