<?php
declare(strict_types=1);

function parseEnvFile(string $filePath): array
{
    if (!is_file($filePath)) {
        return [];
    }

    $vars = [];
    $lines = file($filePath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    if ($lines === false) {
        return [];
    }

    foreach ($lines as $line) {
        $trimmed = trim($line);
        if ($trimmed === '' || str_starts_with($trimmed, '#')) {
            continue;
        }

        $parts = explode('=', $trimmed, 2);
        if (count($parts) !== 2) {
            continue;
        }

        $key = trim($parts[0]);
        $value = trim($parts[1]);

        if (
            (str_starts_with($value, '"') && str_ends_with($value, '"')) ||
            (str_starts_with($value, "'") && str_ends_with($value, "'"))
        ) {
            $value = substr($value, 1, -1);
        }

        $vars[$key] = $value;
    }

    return $vars;
}

function envValue(array $env, string $key, ?string $default = null): ?string
{
    $runtime = getenv($key);
    if ($runtime !== false) {
        return $runtime;
    }

    return $env[$key] ?? $default;
}

function csvToArray(?string $value): array
{
    if ($value === null || trim($value) === '') {
        return [];
    }

    $items = array_map('trim', explode(',', $value));
    return array_values(array_filter($items, static fn ($item) => $item !== ''));
}

$env = parseEnvFile(dirname(__DIR__) . '/.env');

return [
    'app' => [
        'jwt_secret' => envValue($env, 'JWT_SECRET', 'change-this-jwt-secret-in-production'),
        'allowed_origins' => array_values(array_unique(array_filter(array_merge(
            [
                'https://weatherly.co.ke',
                'https://www.weatherly.co.ke',
                'https://windy.weatherly.co.ke',
            ],
            csvToArray(envValue($env, 'FRONTEND_ORIGINS', ''))
        )))),
    ],
    'db' => [
        'host' => envValue($env, 'DB_HOST', 'localhost'),
        'port' => (int) envValue($env, 'DB_PORT', '3306'),
        'name' => envValue($env, 'DB_NAME', 'weatherly_api'),
        'user' => envValue($env, 'DB_USER', 'weatherly_api'),
        'pass' => envValue($env, 'DB_PASS', 'Weatherly2026@#'),
        'charset' => envValue($env, 'DB_CHARSET', 'utf8mb4'),
    ],
    'api_keys' => [
        'checkwx' => envValue($env, 'CHECKWX_API_KEY'),
        'icao' => envValue($env, 'ICAO_API_KEY'),
        'openweather' => envValue($env, 'OPENWEATHER_API_KEY'),
        'windy' => envValue($env, 'WINDY_API_KEY'),
        'avwx' => envValue($env, 'AVWX_API_KEY'),
        'openaip' => envValue($env, 'OPENAIP_API_KEY'),
    ],
    'seed' => [
        'admin_email' => envValue($env, 'ADMIN_EMAIL', 'admin@weatherly.co.ke'),
        'admin_password' => envValue($env, 'ADMIN_PASSWORD', 'Weatherly2026@#'),
        'admin_username' => envValue($env, 'ADMIN_USERNAME', 'System Admin'),
    ],
];

