<?php
declare(strict_types=1);

function defaultPreferences(): array
{
    return [
        'temperatureUnit' => 'celsius',
        'windUnit' => 'kts',
        'darkMode' => false,
        'emailAlerts' => false,
        'smsAlerts' => false,
    ];
}

function safeJsonDecode(?string $value, $fallback)
{
    if ($value === null || $value === '') {
        return $fallback;
    }

    $decoded = json_decode($value, true);
    return json_last_error() === JSON_ERROR_NONE ? $decoded : $fallback;
}

function toPublicUser(array $userRow): ?array
{
    if (empty($userRow['id']) || empty($userRow['email'])) {
        return null;
    }

    $preferences = safeJsonDecode($userRow['preferences'] ?? null, []);
    $favorites = safeJsonDecode($userRow['favorite_airports'] ?? null, []);

    return [
        'id' => $userRow['id'],
        'email' => $userRow['email'],
        'username' => $userRow['username'] ?? explode('@', (string) $userRow['email'])[0] ?? 'Pilot',
        'role' => $userRow['role'] ?? 'user',
        'licenseNumber' => $userRow['license_number'] ?? '',
        'pilotCertification' => $userRow['pilot_certification'] ?? '',
        'favoriteAirports' => is_array($favorites) ? $favorites : [],
        'preferences' => array_merge(defaultPreferences(), is_array($preferences) ? $preferences : []),
        'createdAt' => $userRow['created_at'] ?? null,
    ];
}

function normalizeApiKeyValue($value): ?string
{
    if ($value === null) {
        return null;
    }

    if (is_string($value) && trim($value) === '') {
        return null;
    }

    return (string) $value;
}

function generateUuidV4(): string
{
    $bytes = random_bytes(16);
    $bytes[6] = chr((ord($bytes[6]) & 0x0f) | 0x40);
    $bytes[8] = chr((ord($bytes[8]) & 0x3f) | 0x80);
    return vsprintf('%s%s-%s-%s-%s-%s%s%s', str_split(bin2hex($bytes), 4));
}

function isLocalDevOrigin(string $origin): bool
{
    $host = parse_url($origin, PHP_URL_HOST);
    if (!is_string($host)) {
        return false;
    }

    return in_array($host, ['localhost', '127.0.0.1', '::1'], true);
}

function applyCors(string $origin, array $allowedOrigins): void
{
    $method = strtoupper($_SERVER['REQUEST_METHOD'] ?? 'GET');

    header('Access-Control-Allow-Credentials: true');
    header('Access-Control-Allow-Headers: Content-Type, Authorization');
    header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');

    if ($origin === '') {
        if ($method === 'OPTIONS') {
            http_response_code(204);
            exit;
        }
        return;
    }

    if (isLocalDevOrigin($origin) || in_array($origin, $allowedOrigins, true)) {
        header('Access-Control-Allow-Origin: ' . $origin);
        header('Vary: Origin');
        if ($method === 'OPTIONS') {
            http_response_code(204);
            exit;
        }
        return;
    }

    fail('Not allowed by CORS', 403);
    exit;
}

function jsonBody(): array
{
    $raw = file_get_contents('php://input');
    if ($raw === false || trim($raw) === '') {
        return [];
    }

    $decoded = json_decode($raw, true);
    return is_array($decoded) ? $decoded : [];
}

function resolveApiPath(): string
{
    $requestPath = parse_url($_SERVER['REQUEST_URI'] ?? '/', PHP_URL_PATH) ?: '/';
    $requestPath = str_replace('\\', '/', $requestPath);

    $scriptName = str_replace('\\', '/', $_SERVER['SCRIPT_NAME'] ?? '');
    $scriptDir = rtrim(dirname($scriptName), '/');

    if ($scriptDir !== '' && $scriptDir !== '.' && str_starts_with($requestPath, $scriptDir)) {
        $requestPath = substr($requestPath, strlen($scriptDir));
    }

    if (str_starts_with($requestPath, '/index.php')) {
        $requestPath = substr($requestPath, strlen('/index.php'));
    }

    if (str_starts_with($requestPath, '/api/')) {
        $requestPath = substr($requestPath, 4);
    } elseif ($requestPath === '/api') {
        $requestPath = '/';
    }

    $normalized = '/' . trim($requestPath, '/');
    return $normalized === '//' ? '/' : $normalized;
}
